<?php
namespace WPDM\AddOn\Dropbox;

use WPDM\__\Messages;

class DropboxClient {
	private $clientId;
	private $clientSecret;
	private $accessToken;
	private $refreshToken;

	private $basicAuth;

	public function __construct( $clientId, $clientSecret, $accessToken = null ) {
		$this->clientId     = $clientId;
		$this->clientSecret = $clientSecret;
		$this->accessToken  = $accessToken;
		$this->basicAuth = base64_encode("{$clientId}:{$clientSecret}");
	}

	function isConnected(  ) {
		return !empty( $this->accessToken );
	}

	private function makeRequest( $url, $method = 'POST', $headers = [], $data = null, $decode = true ) {

		$curl = curl_init( $url );
		curl_setopt( $curl, CURLOPT_RETURNTRANSFER, true );
		curl_setopt( $curl, CURLOPT_CUSTOMREQUEST, $method );

		if ( ! empty( $headers ) ) {
			curl_setopt( $curl, CURLOPT_HTTPHEADER, $headers );
		}

		if ( $data ) {
			curl_setopt( $curl, CURLOPT_POSTFIELDS, $data );
		}

		$response = curl_exec( $curl );
		$error    = curl_error( $curl );
		curl_close( $curl );
		//if(substr_count($url, 'temp'))
		//	wpdmdd($response, $error);
		if ( $error ) {

			throw new \Exception( "cURL error: $error" );
		}
		return $decode ? json_decode( $response, true ) : $response;
	}

	public function authorize( $redirectUri ) {
		$url = "https://www.dropbox.com/oauth2/authorize?response_type=code&client_id={$this->clientId}&token_access_type=offline&redirect_uri=" . urlencode( $redirectUri );
		return $url;
	}

	public function getToken( $authorizationCode, $redirectUri ) {
		$url     = "https://api.dropboxapi.com/oauth2/token";
		$headers = [ 'Content-Type: application/x-www-form-urlencoded' ];
		$data    = http_build_query( [
			'code'          => $authorizationCode,
			'grant_type'    => 'authorization_code',
			'client_id'     => $this->clientId,
			'client_secret' => $this->clientSecret,
			'redirect_uri'  => $redirectUri,
		] );

		$response          = $this->makeRequest( $url, 'POST', $headers, $data );
		$this->accessToken = $response['access_token'] ?? null;
		$this->refreshToken = $response['refresh_token'] ?? null;
		return $response;
	}

	public function refreshToken($refreshToken)
	{
		$url = "https://api.dropboxapi.com/oauth2/token";
		$headers = ['Content-Type: application/x-www-form-urlencoded'];
		$data = http_build_query([
			'grant_type' => 'refresh_token',
			'refresh_token' => $refreshToken,
			'client_id' => $this->clientId,
			'client_secret' => $this->clientSecret,
		]);

		$response = $this->makeRequest($url, 'POST', $headers, $data);
		// Update the access token
		$this->accessToken = $response['access_token'] ?? null;
		$this->refreshToken = $refreshToken;
		return $response['access_token'];
	}


	public function download( $path ) {
		$url     = "https://content.dropboxapi.com/2/files/download";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Dropbox-API-Arg: " . json_encode( [ 'path' => $path ] )
		];

		return $this->makeRequest( $url, 'POST', $headers );
	}

	public function downloadZip( $path ) {
		$url     = "https://content.dropboxapi.com/2/files/download_zip";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Dropbox-API-Arg: " . json_encode( [ 'path' => $path ] )
		];

		WPDM()->fileSystem->downloadHeaders(sanitize_file_name( $path ).".zip");
		echo $this->makeRequest( $url, 'POST', $headers, null, false );
		die();
		//return $zipData;
	}

	public function getMetadata( $path ) {
		$url     = "https://api.dropboxapi.com/2/files/get_metadata";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Content-Type: application/json"
		];
		$data    = json_encode( [ 'path' => $path ] );

		return $this->makeRequest( $url, 'POST', $headers, $data );
	}

	public function getPreview( $path ) {
		$url     = "https://content.dropboxapi.com/2/files/get_preview";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Dropbox-API-Arg: " . json_encode( [ 'path' => $path ] )
		];

		return $this->makeRequest( $url, 'POST', $headers );
	}

	public function getTemporaryLink( $path ) {
		$url     = "https://api.dropboxapi.com/2/files/get_temporary_link";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Content-Type: application/json"
		];
		$data    = json_encode( [ 'path' => $path ] );
		$response = $this->makeRequest( $url, 'POST', $headers, $data );
		return $response['link'];
	}

	public function getThumbnail( $path ) {
		$url     = "https://content.dropboxapi.com/2/files/get_thumbnail";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Dropbox-API-Arg: {\"format\":\"jpeg\",\"mode\":\"strict\",\"path\":\"{$path}\",\"quality\":\"quality_80\",\"size\":\"w960h640\"}"
		];

		return $this->makeRequest( $url, 'POST', $headers, null, false );
	}


	public function listFolder( $path = "" ) {
		$url     = "https://api.dropboxapi.com/2/files/list_folder";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Content-Type: application/json"
		];
		if($path === '/') $path = "";
		$data    = '{
				    "path": "'.$path.'", 
				    "recursive": false, 
				    "include_media_info": false, 
				    "include_deleted": false, 
				    "include_has_explicit_shared_members": false, 
				    "include_mounted_folders": false, 
				    "include_non_downloadable_files": false
				}';

		return $this->makeRequest( $url, 'POST', $headers, $data );
	}

	public function upload( $path, $contents ) {
		$url     = "https://content.dropboxapi.com/2/files/upload";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Dropbox-API-Arg: " . json_encode( [ 'path'       => $path,
			                                     'mode'       => 'add',
			                                     'autorename' => true,
			                                     'mute'       => false
			] ),
			"Content-Type: application/octet-stream"
		];

		return $this->makeRequest( $url, 'POST', $headers, $contents );
	}

	public function createFolder( $path ) {
		$url     = "https://api.dropboxapi.com/2/files/create_folder_v2";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Content-Type: application/json"
		];
		$data    = json_encode( [ 'path' => $path ] );

		return $this->makeRequest( $url, 'POST', $headers, $data );
	}

	public function delete( $path ) {
		$url     = "https://api.dropboxapi.com/2/files/delete_v2";
		$headers = [
			"Authorization: Bearer {$this->accessToken}",
			"Content-Type: application/json"
		];
		$data    = json_encode( [ 'path' => $path ] );

		return $this->makeRequest( $url, 'POST', $headers, $data );
	}
}
